<?php
header("Content-Type: application/json");

require_once 'config.php';

// Create connection
$conn = get_db_connection();

// Check connection
if ($conn->connect_error) {
    die(json_encode(['status' => 'error', 'message' => 'Connection failed: ' . $conn->connect_error]));
}

// Retrieve and sanitize POST data
$user_email = isset($_POST['user_email']) ? strtolower(trim($_POST['user_email'])) : '';
$title = isset($_POST['title']) ? trim($_POST['title']) : '';
$message = isset($_POST['message']) ? trim($_POST['message']) : '';

// Log received data for debugging
file_put_contents('debug_log.txt', "Received Email: " . $user_email . "\n", FILE_APPEND);
file_put_contents('debug_log.txt', "Received Title: " . $title . "\n", FILE_APPEND);
file_put_contents('debug_log.txt', "Received Message: " . $message . "\n", FILE_APPEND);

// Check if email is empty
if (empty($user_email)) {
    die(json_encode(['status' => 'error', 'message' => 'Email not received or empty']));
}

// Fetch FCM token from the database
$query = "SELECT fcmToken FROM user_data WHERE LOWER(email) = ?";
$stmt = $conn->prepare($query);
$stmt->bind_param("s", $user_email);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows > 0) {
    $row = $result->fetch_assoc();
    $fcm_token = $row['fcmToken'];

    // Fetch OAuth token from the Google service account
    $access_token = getAccessToken(); // Custom function to get OAuth token

    if (!$access_token) {
        die(json_encode(['status' => 'error', 'message' => 'Failed to get OAuth token']));
    }

    // Prepare FCM message for HTTP v1 API
    $url = 'https://fcm.googleapis.com/v1/projects/gopalganj-96601/messages:send';
    $fields = array(
        'message' => array(
            'token' => $fcm_token,
            'notification' => array(
                'title' => $title,
                'body' => $message
            ),
            'android' => array(  // Optional: Include platform-specific fields
                'priority' => 'high'
            )
        )
    );

    $headers = array(
        'Authorization: Bearer ' . $access_token, // Bearer token from OAuth
        'Content-Type: application/json'
    );

    // Send the notification via cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
    $result = curl_exec($ch);

    // Check if curl_exec() failed
    if ($result === FALSE) {
        echo json_encode(['status' => 'error', 'message' => 'Curl failed: ' . curl_error($ch)]);
    } else {
        // Check the HTTP status code
        $http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        if ($http_status == 200) {
            echo json_encode(['status' => 'success', 'message' => 'Notification sent successfully', 'response' => json_decode($result)]);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Failed with status: ' . $http_status, 'response' => json_decode($result)]);
        }
    }
    curl_close($ch);
} else {
    echo json_encode(['status' => 'error', 'message' => 'Email not found or FCM token is missing']);
}

$stmt->close();
$conn->close();

// Function to fetch OAuth 2.0 access token using the service account key
function getAccessToken() {
    $key_file_location = 'service-account-file.json'; // Path to your service account key file
    $scope = 'https://www.googleapis.com/auth/firebase.messaging';

    // Load the service account key
    $credentials = json_decode(file_get_contents($key_file_location), true);
    $jwt = generateJWT($credentials, $scope);

    // Get OAuth 2.0 token using JWT
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://oauth2.googleapis.com/token');
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query(array(
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion' => $jwt
    )));
    $response = curl_exec($ch);
    curl_close($ch);

    if ($response) {
        $data = json_decode($response, true);
        return $data['access_token'] ?? null;
    }

    return null;
}

// Function to generate a JWT (JSON Web Token)
function generateJWT($credentials, $scope) {
    $header = json_encode([
        'alg' => 'RS256',
        'typ' => 'JWT'
    ]);

    $now = time();
    $claims = json_encode([
        'iss' => $credentials['client_email'],
        'scope' => $scope,
        'aud' => 'https://oauth2.googleapis.com/token',
        'iat' => $now,
        'exp' => $now + 3600
    ]);

    $header = base64UrlEncode($header);
    $claims = base64UrlEncode($claims);

    $signature = '';
    openssl_sign("$header.$claims", $signature, $credentials['private_key'], 'SHA256');

    return "$header.$claims." . base64UrlEncode($signature);
}

// Function to base64 URL encode data
function base64UrlEncode($data) {
    return str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($data));
}
?>
